//
//  ModelController.m
//  Creating a Simple iOS App in Xcode
//
//  Created by Vandad Nahavandipoor on 07/07/2011.
//  Copyright 2011 Pixolity Ltd. All rights reserved.
//

#import "ModelController.h"

#import "DataViewController.h"

/*
 Obiekt kontrolera zarządzający prostym modelem -- zbiorem składającym się z nazw miesięcy.
 
 Kontroler jest źródłem danych dla kontrolera widoku strony; implementuje metody pageViewController:viewControllerBeforeViewController: i pageViewController:viewControllerAfterViewController:.
 Implementuje także własną metodę viewControllerAtIndex:, która jest użyteczna podczas implementacji metod źródła danych oraz w trakcie początkowej konfiguracji aplikacji.
 
 W rzeczywistości, nie ma konieczności wcześniejszego tworzenia kontrolerów widoku dla poszczególnych stron, to stanowi niepotrzebne obciążenie. Na podstawie modelu danych metody odpowiadają za utworzenie, konfigurację i przekazanie nowego kontrolera widoku na żądanie.
 */

@interface ModelController()
@property (readonly, strong, nonatomic) NSArray *pageData;
@end

@implementation ModelController

@synthesize pageData = _pageData;

- (id)init
{
    self = [super init];
    if (self) {
    // Utworzenie modelu danych.
    NSDateFormatter *dateFormatter = [[NSDateFormatter alloc] init];
    _pageData = [[dateFormatter monthSymbols] copy];
    }
    return self;
}

- (DataViewController *)viewControllerAtIndex:(NSUInteger)index
{   
    // Zwrócenie danych kontrolera widoku dla wskazanego indeksu.
    if (([self.pageData count] == 0) || (index >= [self.pageData count])) {
        return nil;
    }
    
    // Utworzenie nowego kontrolera widoku i przekazanie mu odpowiednich danych.
    DataViewController *dataViewController;
    if ([[UIDevice currentDevice] userInterfaceIdiom] == UIUserInterfaceIdiomPhone) {
        dataViewController = [[DataViewController alloc] initWithNibName:@"DataViewController_iPhone" bundle:nil];
    } else {
        dataViewController = [[DataViewController alloc] initWithNibName:@"DataViewController_iPad" bundle:nil];
    }
    dataViewController.dataObject = [self.pageData objectAtIndex:index];
    return dataViewController;
}

- (NSUInteger)indexOfViewController:(DataViewController *)viewController
{   
    /*
     Zwrot indeksu kontrolera widoku wskazanych danych.
     W celu zachowania prostoty przedstawiona implementacja wykorzystuje statyczną tablicę obiektów modelu, a kontroler widoku przechowuje obiekt modelu. W ten sposób obiekt modelu można wykorzystać do identyfikacji indeksu.
     */
    return [self.pageData indexOfObject:viewController.dataObject];
}

#pragma mark - Page View Controller Data Source

- (UIViewController *)pageViewController:(UIPageViewController *)pageViewController viewControllerBeforeViewController:(UIViewController *)viewController
{
    NSUInteger index = [self indexOfViewController:(DataViewController *)viewController];
    if ((index == 0) || (index == NSNotFound)) {
        return nil;
    }
    
    index--;
    return [self viewControllerAtIndex:index];
}

- (UIViewController *)pageViewController:(UIPageViewController *)pageViewController viewControllerAfterViewController:(UIViewController *)viewController
{
    NSUInteger index = [self indexOfViewController:(DataViewController *)viewController];
    if (index == NSNotFound) {
        return nil;
    }
    
    index++;
    if (index == [self.pageData count]) {
        return nil;
    }
    return [self viewControllerAtIndex:index];
}

@end
